# Helpful function to throw errors (If looks too cumbersome in my opinion)
assert <- function(condition, message) {
  if (!condition) {
    # warning(message)
  }
}

#' Loads a graph from Node / Edge CSV files
#'
#' This function takes the file paths to two CSV files: one containing the node information of a network
#' and the other containing the edge information of the network. It returns a list containing several
#' useful data related to the network.
#' @param nodeinfo_path Path to a CSV file containing the node information. This CSV file should follow the convention of `node_name, initial_value`.
#' @param edgeinfo_path Path to a CSV file containing the edge information. This CSV file should follow the convention of `source_node_name, target_node_name, edge_weight`.
#' @param ... Additional arguments passed to read.csv
#' @return A list `L` with the following entries:
#' \item{\code{M:}}{ the adjacency matrix of the network.}
#' \item{\code{initial_values:}}{ an array of initial values for the network.}
#' \item{\code{node_index:}}{ a function that takes a node name and returns its index value.}
#' \item{\code{node_value:}}{ a function that takes a node name and returns its initial value.}
#' \item{\code{edge_index:}}{ a  function that takes two node names and returns the index value of the edge connecting them.}
#' \item{\code{edge_weight:}}{ a function that takes two node names and returns the weight of the edge connecting them.}
#' \item{\code{g:}}{ the igraph graph associated with the network. }
#' \item{\code{edges:}}{ an array of edges.}
#' \item{\code{nodes:}}{ an array of nodes.}
#' \item{\code{node_names:}}{ an array of node names.}
#' @import igraph
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    M <- graph.from.csv(node_file, edge_file)
#' }
graph.from.csv <- function(nodeinfo_path, edgeinfo_path, ...)
{
  # Load the data from CSV files
  nodes_data <- utils::read.csv(nodeinfo_path, ...)
  edges_data <- utils::read.csv(edgeinfo_path, ...)

  if(any(duplicated(nodes_data[,1])))
  {
    warning("WARNING: Duplicate entries in the nodes data. This should never happen!")
  }
  # Create an empty graph
  g <- igraph::graph.empty()

  # Get the names and initial values from the data
  node_names <- nodes_data[, 1]
  node_values <- nodes_data[, 2]

  # Add the nodes to the graph
  for(i in 1:length(node_names)){
    name <- node_names[i]
    value <- node_values[i]

    g <- igraph::add_vertices(g, nv = 1, attr = list("name" = name, "value" = value))
  }

  # Get the source, target and weight of edges
  edge_sources <- edges_data[, 1]
  edge_targets <- edges_data[, 2]
  edge_weights <- edges_data[, 3]

  # Add the edges to the graph
  len <- length(edge_sources)
  for(i in 1:len){
    source <- edge_sources[i]
    target <- edge_targets[i]
    weight <- edge_weights[i]

    g <- igraph::add_edges(g, c(source, target), attr = list("weight" = weight))
  }

  # A function that takes a node name and returns the node id
  node_index <- function(name){
    idx <- which(V(g)$name == name)
    return(idx)
  }
  # A function that takes a node name and returns the initial value of the node
  node_value <- function(name){
    idx <- which(igraph::V(g)$name == name)
    x0 <- igraph::V(g)$value[idx]
    return(x0)
  }
  # A function that takes two node names, and returns the index of the edge connecting them
  edge_index <- function(source, target){
    ids <- which(igraph::V(g)$name == source)
    idt <- which(igraph::V(g)$name == target)
    return(c(ids, idt))
  }
  # A function that takes two node names, and returns the value of the edge connecting them
  edge_weight <- function(source, target){
    idx <- igraph::get.edge.ids(g, c(source, target))
    weight <- igraph::E(g)$weight[idx]
    return(weight)
  }
  # The adjacency matrix of the network
  M <- as.matrix(igraph::as_adjacency_matrix(g, attr="weight", names = FALSE))
  # The array of initial values of the nodes
  initial_values <- igraph::V(g)$value

  # List of nodes
  nodes <- lapply(1:length(igraph::V(g)), function(i) i)

  # List of edges
  edge_list <- igraph::as_edgelist(g)
  edges <- lapply(1:nrow(edge_list), function(i) {
    source_name = edge_list[i, 1]
    target_name = edge_list[i, 2]
    source_idx = which(igraph::V(g)$name == source_name)
    target_idx = which(igraph::V(g)$name == target_name)
    return(c(source_idx, target_idx))
  })
  output <- list("M" = M,                            # Adjacency matrix of the graph.
                 "initial_values" = initial_values,  # Array of initial values.
                 "node_index" = node_index,          # Function that takes a node name, and outputs its index value.
                 "node_value" = node_value,          # Function that takes a node name, and output its initial value.
                 "edge_index" = edge_index,          # Function that takes two node names, and outputs the id of the edge connecting them.
                 "edge_weight" = edge_weight,        # Function that takes twi nodw names, and outputs the value of the edge connecting them.
                 "g" = g,                            # The graph as an igraph object
                 "edges" = edges,                    # Array of edges
                 "nodes" = nodes,                    # Array of nodes
                 "node_names" = node_names)
  return(output)
}
