% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mean_amr_distance.R
\name{mean_amr_distance}
\alias{mean_amr_distance}
\alias{mean_amr_distance.sir}
\alias{mean_amr_distance.data.frame}
\alias{amr_distance_from_row}
\title{Calculate the Mean AMR Distance}
\usage{
mean_amr_distance(x, ...)

\method{mean_amr_distance}{sir}(x, ..., combine_SI = TRUE)

\method{mean_amr_distance}{data.frame}(x, ..., combine_SI = TRUE)

amr_distance_from_row(amr_distance, row)
}
\arguments{
\item{x}{A vector of class \link[=as.sir]{sir}, \link[=as.mic]{mic} or \link[=as.disk]{disk}, or a \link{data.frame} containing columns of any of these classes.}

\item{...}{Variables to select. Supports \link[tidyselect:starts_with]{tidyselect language} such as \code{where(is.mic)}, \code{starts_with(...)}, or \code{column1:column4}, and can thus also be \link[=amr_selector]{antimicrobial selectors}.}

\item{combine_SI}{A \link{logical} to indicate whether all values of S, SDD, and I must be merged into one, so the input only consists of S+I vs. R (susceptible vs. resistant) - the default is \code{TRUE}.}

\item{amr_distance}{The outcome of \code{\link[=mean_amr_distance]{mean_amr_distance()}}.}

\item{row}{An index, such as a row number.}
}
\description{
Calculates a normalised mean for antimicrobial resistance between multiple observations, to help to identify similar isolates without comparing antibiograms by hand.
}
\details{
The mean AMR distance is effectively \href{https://en.wikipedia.org/wiki/Standard_score}{the Z-score}; a normalised numeric value to compare AMR test results which can help to identify similar isolates, without comparing antibiograms by hand.

MIC values (see \code{\link[=as.mic]{as.mic()}}) are transformed with \code{\link[=log2]{log2()}} first; their distance is thus calculated as \code{(log2(x) - mean(log2(x))) / sd(log2(x))}.

SIR values (see \code{\link[=as.sir]{as.sir()}}) are transformed using \code{"S"} = 1, \code{"I"} = 2, and \code{"R"} = 3. If \code{combine_SI} is \code{TRUE} (default), the \code{"I"} will be considered to be 1.

For data sets, the mean AMR distance will be calculated per column, after which the mean per row will be returned, see \emph{Examples}.

Use \code{\link[=amr_distance_from_row]{amr_distance_from_row()}} to subtract distances from the distance of one row, see \emph{Examples}.
}
\section{Interpretation}{

Isolates with distances less than 0.01 difference from each other should be considered similar. Differences lower than 0.025 should be considered suspicious.
}

\examples{
sir <- random_sir(10)
sir
mean_amr_distance(sir)

mic <- random_mic(10)
mic
mean_amr_distance(mic)
# equal to the Z-score of their log2:
(log2(mic) - mean(log2(mic))) / sd(log2(mic))

disk <- random_disk(10)
disk
mean_amr_distance(disk)

y <- data.frame(
  id = LETTERS[1:10],
  amox = random_sir(10, ab = "amox", mo = "Escherichia coli"),
  cipr = random_disk(10, ab = "cipr", mo = "Escherichia coli"),
  gent = random_mic(10, ab = "gent", mo = "Escherichia coli"),
  tobr = random_mic(10, ab = "tobr", mo = "Escherichia coli")
)
y
mean_amr_distance(y)
y$amr_distance <- mean_amr_distance(y, is.mic(y))
y[order(y$amr_distance), ]

if (require("dplyr")) {
  y \%>\%
    mutate(
      amr_distance = mean_amr_distance(y),
      check_id_C = amr_distance_from_row(amr_distance, id == "C")
    ) \%>\%
    arrange(check_id_C)
}
if (require("dplyr")) {
  # support for groups
  example_isolates \%>\%
    filter(mo_genus() == "Enterococcus" & mo_species() != "") \%>\%
    select(mo, TCY, carbapenems()) \%>\%
    group_by(mo) \%>\%
    mutate(dist = mean_amr_distance(.)) \%>\%
    arrange(mo, dist)
}
}
